; Child.asm - demonstrates child windows
;
; Child windows are windows within a window, and have their own window procedure.
; They can be useful for dividing up a main window into different sections. With
; this technique they can respond to messages sent to themselves and then either
; process them directly or could pass them up up the main window. In the latter
; case, the child windows essentially become input devices. For example, you
; could divide up a window into a series of squares and whenever a mouse is
; clicked in a square send a message to the main window. This technique could be
; used for a minesweeper type game, where a child window merely tells the main
; window the coords of a clicked square, and the main window can tell the child
; windows to repaint as required to show their status.

%define _WINMESSAGES_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

ddglobal _gbl_hInstance

procglobal WinMain, hInstance, hPrevInstance, lpszCmdLine, nCmdShow
	ddlocal		_hwnd
	struclocal	_wndclass, WNDCLASSEX, _msg, MSG
	endlocals
	WinMainPrologue
	mov	eax, .hInstance
	mov	._gbl_hInstance, eax
	mov	esi, ._wndclass
	mov	edi, ._msg
	mov	[esi + WNDCLASSEX.cbSize], dword WNDCLASSEX_size
	mov	[esi + WNDCLASSEX.style], dword CS_HREDRAW | CS_VREDRAW
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _WndProc
	mov	[esi + WNDCLASSEX.cbClsExtra], dword 0
	mov	[esi + WNDCLASSEX.cbWndExtra], dword 0
	mov	eax, .hInstance
	mov	[esi + WNDCLASSEX.hInstance], eax
	sc LoadIcon, NULL, IDI_APPLICATION
	mov	[esi + WNDCLASSEX.hIcon], eax
	sc LoadCursor, NULL, IDC_ARROW
	mov	[esi + WNDCLASSEX.hCursor], eax
	sc GetStockObject, WHITE_BRUSH
	mov	[esi + WNDCLASSEX.hbrBackground], eax
	mov	[esi + WNDCLASSEX.lpszMenuName], dword NULL
	TEXTlocal szClassName, 'MyClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword .szClassName
	sc RegisterClassEx, esi
	cmp	eax, TRUE
	je	near _WinMain_Fail
	;
	; Having created the main window class, we need to create a window class
	; for the child windows. The only changes we need to make are the window
	; procedure address and the class name. Note that we use TEXTglobal
	; as we need to refer to the class name in the window procedure
	;
	mov	[esi + WNDCLASSEX.lpfnWndProc], dword _ChildWndProc
	TEXTglobal szChildClassName, 'MyChildClass',0
	mov	[esi + WNDCLASSEX.lpszClassName], dword szChildClassName
	;
	; With these changes made, register the new class
	;
	sc RegisterClassEx, esi
	cmp	eax,TRUE
	je	near _WinMain_Fail
	;
	; Success, so continue to create the main window as normal, As soon as
	; the WM_CREATE message is sent to the window procedure, it will
	; create the child windows automatically.
	;
	TEXTlocal szWndCaption, 'Child windows',0
	sc CreateWindowEx, 0, .szClassName, .szWndCaption, WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, .hInstance, NULL
	mov	._hwnd, eax
	sc ShowWindow, ._hwnd, .nCmdShow
	sc UpdateWindow, ._hwnd
_WinMain_Loop:
	sc GetMessage, ._msg, NULL, 0, 0
	cmp	eax, TRUE
	jne	_WinMain_Loop_End
	sc TranslateMessage, ._msg
	sc DispatchMessage, ._msg
	jmp	_WinMain_Loop
_WinMain_Loop_End:
	mov	eax, [edi + MSG.wParam]
	jmp	_WinMain_End
_WinMain_Fail:
	TEXTlocal szErrorMsg, 'Failed to register window class!',0
	sc MessageBox, NULL, .szErrorMsg, .szWndCaption, MB_ICONERROR
_WinMain_End:
	WinMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
proc _WndProc, hwnd, message, wParam, lParam
	ddstatic	_hwnd1, _hwnd2
	endlocals
	;
	CallbackPrologue
	;
	switch .message
		case WM_CREATE
			;
			; Window being created?
			;
			; Yes, so create the first child window, note that the parameters are
			; different to what we would normally use:
			;
			;  1 - Extended window style, no change
			;  2 - Class name, this changes to the child window class
			;  3 - Caption, set to NULL as there's no point having a caption
			;  4 - Window style, changed to make the window a child window
			;  5-9 - Horizontal, vertical positions, with and height all set to 0
			;        as we'll set these when the main window is sized. This ensures
			;        the child windows change shape whenever the window does and
			;        avoids duplicating code
			;  10 - Parent window handle, set to the main window's handle rather than NULL
			;  11 - In a normal window, this would be the menu identifier, but as this is
			;       a child window, it is used to specify the child window's ID. I've
			;       chosen to call them 1 and 2 respectively.
			;  12 - Instance handle, no change
			;  13 - Extra parameters, no change as we have none
			;
			sc CreateWindowEx, 0, szChildClassName, NULL, WS_CHILDWINDOW | WS_VISIBLE, 0, 0, 0, 0, .hwnd, 1, ._gbl_hInstance, NULL
			;
			; Having created the child window, we store it's window handle
			;
			mov	._hwnd1, eax
			;
			; And now create the second child window
			;
			sc CreateWindowEx, 0, szChildClassName, NULL, WS_CHILDWINDOW | WS_VISIBLE, 0, 0, 0, 0, .hwnd, 2, ._gbl_hInstance, NULL
			;
			; And store its window handle
			;
			mov	._hwnd2, eax
			xor	eax, eax
			break
		case WM_SIZE
			;
			; Window resized?
			;
			; Yes, so get the window's new width and height in esi and edi.
			; In a WM_SIZE message, these are passed in lParam, with
			; the width being in the loword and the height being in the hiword
			;
			mov	eax, .lParam
			movsx	esi, ax
			shr	eax, 16
			movsx	edi, ax
			;
			; Divide the width by two since we have two windows to share this space
			;
			shr	esi, 1
			;
			; Now resize the child windows, via MoveWindow which takes six parameters:
			;
			;   1. Window handle of window to move
			;   2. New horizontal position
			;   3. New vertical position
			;   4. New width
			;   5. New height
			;   6. Send WM_PAINT message?
			;
			sc MoveWindow, ._hwnd1, 0, 0, esi, edi, TRUE
			sc MoveWindow, ._hwnd2, esi, 0, esi, edi, TRUE
			;
			; Note the second resize starts the second window from midway across the
			; main window and I've told Windows to send WM_PAINT messages to both
			; child windows. Note that MoveWindow actually sends a few other messages
			; (movement related ones) which could be useful.
			;
			; If we process the WM_SIZE message, we should return zero according to
			; the API reference. (Like for most messages, unless there's an error)
			;
			xor	eax, eax
			break
		case WM_DESTROY
			sc PostQuitMessage, 0
			xor	eax,eax
			break
		default
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc
;
;-----------------------------------------------------------------------
;
; The child window procedure. This is identical to the way the normal window
; procedure works, ie processing messages, taking the same parameters. Note
; though that we don't need to process WM_DESTROY messages as we won't
; actually get them.
;
proc _ChildWndProc, hwnd, message, wParam, lParam
	ddlocal		_hdc
	struclocal	_rect, RECT, _ps, PAINTSTRUCT
	endlocals
	;
	CallbackPrologue
	;
	switch .message
		case WM_PAINT
			;
			; Repaint window?
			;
			; Yes, first work out which child window we're processing. We do
			; this by checking the window ID, which was set to 1 or 2 when we
			; created the child window. We can obtain this with GetWindowLong,
			; which takes two parameters:
			;
			;   1. Window handle of the window to query
			;   2. Index of attribute to retrieve, see the API reference for
			;      the full list
			;
			; We do this so we can display a different string in each window
			;
			sc GetWindowLong, .hwnd, GWL_ID
			;
			; Esi will point to the string to display. Assume it's child window
			; 1 we're processing to start with
			;
			TEXTlocal _szChild1, 'Child window 1',0
			TEXTlocal _szChild2, 'Child window 2',0
			mov	esi, ._szChild1
			if eax, ne, 1
				mov	esi, ._szChild2
			endif
			;
			; Draw the string onto the window
			;
			sc BeginPaint, .hwnd, ._ps
			mov	._hdc, eax
			sc GetClientRect, .hwnd, ._rect
			sc DrawText, ._hdc, esi, -1, ._rect, DT_SINGLELINE | DT_CENTER | DT_VCENTER
			sc EndPaint, .hwnd, ._ps
			xor	eax,eax
			break
		default
			sc DefWindowProc, .hwnd, .message, .wParam, .lParam
	switchend
	;
	CallbackEpilogue
endproc

[section .bss]

[section .data]
